/*
 * @(#)RelationshipFigure.java  1.1.1  2007-02-14
 *
 * Copyright (c) 2006-2007 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.rer.diagram;

import ch.hslu.cm.rer.model.*;
import ch.hslu.cm.simulation.*;
import ch.hslu.cm.*;
import java.beans.*;
import java.awt.*;
import java.awt.event.*;
import java.io.IOException;
import java.util.*;
import javax.swing.undo.*;
import javax.swing.*;
import java.awt.geom.*;
import org.jhotdraw.app.action.ActionUtil;
import static org.jhotdraw.draw.AttributeKeys.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.decoration.GeneralPathTip;
import org.jhotdraw.draw.decoration.LineDecoration;
import org.jhotdraw.draw.event.FigureAdapter;
import org.jhotdraw.draw.event.FigureEvent;
import org.jhotdraw.draw.layouter.LocatorLayouter;
import org.jhotdraw.draw.liner.ElbowLiner;
import org.jhotdraw.draw.liner.Liner;
import org.jhotdraw.draw.liner.SlantedLiner;
import org.jhotdraw.draw.locator.BezierLabelLocator;
import org.jhotdraw.geom.Geom;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;

/**
 * RelationshipFigure.
 *
 *
 * @author Werner Randelshofer
 * @version 1.1.1 2007-02-14 A Relationship figure can never be filled.
 * <br>1.1 2006-02-21 Support for conditional cardinality added.
 * <br>1.0 2006-01-18 Created.
 */
public class RelationshipFigure extends LabeledLineConnectionFigure
        implements DiagramFigure, PropertyChangeListener, SimulatedObjectListener {

    private Relationship model;
    private BasicStroke shoulderStroke = new BasicStroke(5);

    private class CardinalityAction extends AbstractAction {

        private int cardinality;
        private boolean changeEnd;

        public CardinalityAction(String name, int cardinality, boolean changeEnd) {
            super(name);
            putValue(ActionUtil.SELECTED_KEY,
                    cardinality == ((changeEnd) ? model.getEndCardinality() : model.getStartCardinality()));
            this.cardinality = cardinality;
            this.changeEnd = changeEnd;
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            final int oldValue;
            if (changeEnd) {
                oldValue = model.getEndCardinality();
                model.setEndCardinality(cardinality);
            } else {
                oldValue = model.getStartCardinality();
                model.setStartCardinality(cardinality);
            }

            fireUndoableEditHappened(new AbstractUndoableEdit() {

                @Override
                public String getPresentationName() {
                    ResourceBundleUtil labels = RelationalERModel.labels;
                    return labels.getString("cardinality");
                }

                @Override
                public void undo() throws CannotUndoException {
                    super.undo();
                    if (changeEnd) {
                        model.setEndCardinality(oldValue);
                    } else {
                        model.setStartCardinality(oldValue);
                    }
                }

                @Override
                public void redo() throws CannotUndoException {
                    super.redo();
                    if (changeEnd) {
                        model.setEndCardinality(cardinality);
                    } else {
                        model.setStartCardinality(cardinality);
                    }
                }
            });
        }
    }
// Relations have no arrow tip
    /*
    private static LineDecoration traversableTip = new ArrowTip(Math.PI / 11, 13, 0, false, true, true);
    private static LineDecoration nonTraversableTip;
    static {
    GeneralPath p = new GeneralPath();
    p.moveTo(-4, 8);
    p.lineTo(4, 16);
    p.moveTo(4,  8);
    p.lineTo(-4, 16);
    nonTraversableTip = new GeneralPathTip(p, 0, false, true, true);
    }*/
    private static LineDecoration traversableTip = null;
    private static LineDecoration nonTraversableTip = null;
    private static LineDecoration connectedWithAttributeTip;

    static {
        Path2D.Double p = new Path2D.Double();
        p.moveTo(-2, 0);
        p.lineTo(2, 0);
        p.lineTo(2, 12);
        p.lineTo(-2, 12);
        p.lineTo(-2, 0);
        connectedWithAttributeTip = new GeneralPathTip(p, 0, true, false, true);
    }

    /**
     * This adapter is used, to connect a TextFigure for an attribute with
     * the SimulatedObject model.
     */
    private class LabelAdapter extends FigureAdapter {

        private boolean owner;

        public LabelAdapter(boolean owner) {
            this.owner = owner;
        }

        @Override
        public void attributeChanged(FigureEvent evt) {
            if (evt.getAttribute().equals(TEXT)) {
                String newValue = (String) evt.getNewValue();
                if (owner) {
                    if (newValue == null || newValue.length() == 0) {
                        model.setStartTraversable(false);
                    } else {
                        model.setStartLabel(newValue);
                    }
                } else {
                    if (newValue == null || newValue.length() == 0) {
                        model.setEndTraversable(false);
                    } else {
                        model.setEndLabel(newValue);
                    }
                }
            }
        }
    }

    /** Creates a new instance. */
    public RelationshipFigure() {
        setLayouter(new LocatorLayouter());
        setLiner(new ElbowLiner());
        setModel(new Relationship());

        FILL_COLOR.set(this, null);

        setAttributeEnabled(END_DECORATION, false);
        setAttributeEnabled(START_DECORATION, false);
        setAttributeEnabled(STROKE_DASHES, false);
        //  setAttributeEnabled(FONT_BOLD, false);
        setAttributeEnabled(STROKE_TYPE, false);
        setAttributeEnabled(FONT_ITALIC, false);
        setAttributeEnabled(FONT_UNDERLINE, false);
        setAttributeEnabled(FILL_COLOR, false);
    }

    @Override
    public Relationship getModel() {
        return model;
    }

    public void setModel(Relationship m) {
        if (model != null) {
            model.removePropertyChangeListener(this);
            model.removeSimulatedObjectListener(this);
        }
        model = m;
        if (model != null) {
            model.addPropertyChangeListener(this);
            model.addSimulatedObjectListener(this);
        }
        updateLabels();
    }

    @Override
    public void addNotify(Drawing drawing) {
        super.addNotify(drawing);
        if ((drawing instanceof Diagram) && getModel() != null) {
            getSimulation().add(getModel());
        }
    }

    @Override
    public void removeNotify(Drawing drawing) {
        if (getDrawing() != null && getModel() != null) {
            getSimulation().remove(getModel());
        }
        super.removeNotify(drawing);
    }

    @Override
    public Rectangle2D.Double getDrawingArea() {
        Rectangle2D.Double r = super.getDrawingArea();
        if (shoulderStroke != null) {
            double grow = shoulderStroke.getLineWidth();
            Geom.grow(r, grow, grow);
        }
        return r;
    }

    private void updateLabels() {
        ResourceBundleUtil labels = RelationalERModel.labels;

        willChange();
        basicRemoveAllChildren();

        setAttributeEnabled(START_DECORATION, true);
        setAttributeEnabled(END_DECORATION, true);

        if (model == null) {
            START_DECORATION.set(this, null);
            END_DECORATION.set(this, null);
        } else {
            if (model.getStartCardinality() != 0) {
                TextFigure startCardinality = new TextFigure();
                LocatorLayouter.LAYOUT_LOCATOR.set(startCardinality,
                        new BezierLabelLocator(0, -15d / 180d * Math.PI, 8));
                startCardinality.setEditable(false);
                switch (model.getStartCardinality()) {
                    case Relationship.MANY_CARDINALITY:
                        startCardinality.setText(labels.getString("cardinalityM"));
                        break;
                    case Relationship.CONDITIONAL_CARDINALITY:
                        startCardinality.setText(labels.getString("cardinalityC"));
                        break;
                    case Relationship.MANY_CONDITIONAL_CARDINALITY:
                        startCardinality.setText(labels.getString("cardinalityMC"));
                        break;
                    case Relationship.UNSPECIFIED_CARDINALITY:
                        startCardinality.setText("");
                        break;
                    default:
                        startCardinality.setText(Integer.toString(model.getStartCardinality()));
                        break;
                }
                basicAdd(startCardinality);
            }
            if (model.getEndCardinality() != 0) {
                TextFigure endCardinality = new TextFigure();
                LocatorLayouter.LAYOUT_LOCATOR.set(endCardinality,
                        new BezierLabelLocator(1, -165d / 180d * Math.PI, 8));
                endCardinality.setEditable(false);
                switch (model.getEndCardinality()) {
                    case Relationship.MANY_CARDINALITY:
                        endCardinality.setText(labels.getString("cardinalityM"));
                        break;
                    case Relationship.CONDITIONAL_CARDINALITY:
                        endCardinality.setText(labels.getString("cardinalityC"));
                        break;
                    case Relationship.MANY_CONDITIONAL_CARDINALITY:
                        endCardinality.setText(labels.getString("cardinalityMC"));
                        break;
                    case Relationship.UNSPECIFIED_CARDINALITY:
                        endCardinality.setText("");
                        break;
                    default:
                        endCardinality.setText(Integer.toString(model.getEndCardinality()));
                        break;
                }
                basicAdd(endCardinality);
            }

            if (model.isStartTraversable()) {
                TextFigure ownerLabel = new TextFigure(model.getStartLabel());
                LocatorLayouter.LAYOUT_LOCATOR.set(ownerLabel,
                        new BezierLabelLocator(0.5, -90d / 180d * Math.PI, 2));
                ownerLabel.addFigureListener(new LabelAdapter(true));
                //FONT_BOLD.set(ownerLabel, true);
                basicAdd(ownerLabel);
            }

            if (model.isEndTraversable()) {
                TextFigure memberLabel = new TextFigure(model.getEndLabel());
                LocatorLayouter.LAYOUT_LOCATOR.set(memberLabel,
                        new BezierLabelLocator(0.5, 90d / 180d * Math.PI, 2));
                memberLabel.addFigureListener(new LabelAdapter(false));
                // FONT_BOLD.set(memberLabel, true);
                basicAdd(memberLabel);
            }

            if (model.getEndAttribute() != null) {
                END_DECORATION.set(this, connectedWithAttributeTip);
            } else {
                END_DECORATION.set(this, nonTraversableTip);
            }
            if (model.getStartAttribute() != null) {
                START_DECORATION.set(this, connectedWithAttributeTip);
            } else {
                START_DECORATION.set(this, nonTraversableTip);
            }
        }

        for (Figure f : getChildren()) {
            AbstractAttributedFigure label = (AbstractAttributedFigure) f;
            label.setAttributeEnabled(STROKE_COLOR, false);
            label.setAttributeEnabled(STROKE_WIDTH, false);
            label.setAttributeEnabled(STROKE_DASHES, false);
            label.setAttributeEnabled(STROKE_MITER_LIMIT, false);
            label.setAttributeEnabled(STROKE_JOIN, false);
            label.setAttributeEnabled(FONT_BOLD, false);
            label.set(TEXT_COLOR, TEXT_COLOR.get(this));
            label.set(FONT_FACE, FONT_FACE.get(this));
            label.set(FONT_ITALIC, FONT_ITALIC.get(this));
            label.set(FONT_UNDERLINE, FONT_UNDERLINE.get(this));
            label.set(FONT_SIZE, FONT_SIZE.get(this));
        }

        setAttributeEnabled(START_DECORATION, false);
        setAttributeEnabled(END_DECORATION, false);

        setAttributeEnabled(STROKE_DASHES, true);
        STROKE_DASHES.set(this, (model.isResolved()) ? new double[]{5, 5} : null);
        setAttributeEnabled(STROKE_DASHES, false);


        layout();
        changed();
    }

    @Override
    public Collection<Action> getActions(Point2D.Double p) {
        LinkedList<Action> actions = new LinkedList<Action>();
        Action action;
        final ResourceBundleUtil labels = RelationalERModel.labels;

        Figure item;

        // Changing the cardinality at the start of the connection
        action = new CardinalityAction(labels.getString("cardinality1"), 1, false);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getStartEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityStart");
        actions.add(action);

        action = new CardinalityAction(labels.getString("cardinalityM"), Relationship.MANY_CARDINALITY, false);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getStartEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityStart");
        actions.add(action);

        action = new CardinalityAction(labels.getString("cardinalityC"), Relationship.CONDITIONAL_CARDINALITY, false);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getStartEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityStart");
        actions.add(action);

        action = new CardinalityAction(labels.getString("cardinalityMC"), Relationship.MANY_CONDITIONAL_CARDINALITY, false);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getStartEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityStart");
        actions.add(action);

        action = new CardinalityAction(labels.getString("cardinalityUnspecified"), Relationship.UNSPECIFIED_CARDINALITY, false);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getStartEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityStart");
        actions.add(action);

        // Changing the cardinality at the end of the connection
        action = new CardinalityAction(labels.getString("cardinality1"), 1, true);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getEndEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityEnd");
        actions.add(action);

        action = new CardinalityAction(labels.getString("cardinalityM"), Relationship.MANY_CARDINALITY, true);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getEndEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityEnd");
        actions.add(action);

        action = new CardinalityAction(labels.getString("cardinalityC"), Relationship.CONDITIONAL_CARDINALITY, true);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getEndEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityEnd");
        actions.add(action);

        action = new CardinalityAction(labels.getString("cardinalityMC"), Relationship.MANY_CONDITIONAL_CARDINALITY, true);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getEndEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityEnd");
        actions.add(action);

        action = new CardinalityAction(labels.getString("cardinalityUnspecified"), Relationship.UNSPECIFIED_CARDINALITY, true);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("cardinalityAt", model.getEndEntity().getName()));
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinalityEnd");
        actions.add(action);

        action = new AbstractAction(model.isStartTraversable() ? labels.getString("relationshipStartLabelHide") : labels.getString("relationshipStartLabelShow")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setStartTraversable(!model.isStartTraversable());
            }
        };

        actions.add(action);
        action = new AbstractAction(model.isEndTraversable() ? labels.getString("relationshipEndLabelHide") : labels.getString("relationshipEndLabelShow")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setEndTraversable(!model.isEndTraversable());
            }
        };
        actions.add(action);

        action = new AbstractAction(labels.getString(model.isResolved() ? "relationshipDontMarkAsResolved" : "relationshipMarkAsResolved")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setResolved(!model.isResolved());
                fireUndoableEditHappened(new AbstractUndoableEdit() {

                    @Override
                    public String getPresentationName() {
                        return labels.getString("relationship");
                    }

                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        model.setResolved(!model.isResolved());
                    }

                    @Override
                    public void redo() throws CannotUndoException {
                        super.undo();
                        model.setResolved(!model.isResolved());
                    }
                });
            }
        };
        actions.add(action);

        action = new AbstractAction(labels.getString("relationshipReverse")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                reverseConnection();
                fireUndoableEditHappened(new AbstractUndoableEdit() {

                    @Override
                    public String getPresentationName() {
                        return labels.getString("cardinality");
                    }

                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        reverseConnection();
                    }

                    @Override
                    public void redo() throws CannotUndoException {
                        super.undo();
                        reverseConnection();
                    }
                });
            }
        };
        actions.add(action);

        action = new AbstractAction(labels.getString("linerManual")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                final Liner oldValue = getLiner();
                willChange();
                setLiner(null);
                changed();
                fireFigureHandlesChanged();
                fireUndoableEditHappened(new AbstractUndoableEdit() {

                    @Override
                    public String getPresentationName() {
                        return labels.getString("liner");
                    }

                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        willChange();
                        setLiner(oldValue);
                        changed();
                        fireFigureHandlesChanged();
                    }

                    @Override
                    public void redo() throws CannotUndoException {
                        super.undo();
                        willChange();
                        setLiner(null);
                        changed();
                        fireFigureHandlesChanged();
                    }
                });
            }
        };
        action.putValue(ActionUtil.SELECTED_KEY, getLiner() == null);
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "liner");
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("liner"));
        actions.add(action);

        action = new AbstractAction(labels.getString("linerSlanted")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                final Liner oldValue = getLiner();
                final Liner newValue = new SlantedLiner();
                willChange();
                setLiner(newValue);
                changed();
                fireFigureHandlesChanged();
                fireUndoableEditHappened(new AbstractUndoableEdit() {

                    @Override
                    public String getPresentationName() {
                        return labels.getString("liner");
                    }

                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        willChange();
                        setLiner(oldValue);
                        changed();
                        fireFigureHandlesChanged();
                    }

                    @Override
                    public void redo() throws CannotUndoException {
                        super.undo();
                        willChange();
                        setLiner(newValue);
                        changed();
                        fireFigureHandlesChanged();
                    }
                });
            }
        };
        action.putValue(ActionUtil.SELECTED_KEY, getLiner() instanceof SlantedLiner);
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "liner");
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("liner"));
        actions.add(action);

        action = new AbstractAction(labels.getString("linerElbow")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                final Liner oldValue = getLiner();
                final Liner newValue = new ElbowLiner();
                willChange();
                setLiner(newValue);
                changed();
                fireUndoableEditHappened(new AbstractUndoableEdit() {

                    @Override
                    public String getPresentationName() {
                        return labels.getString("liner");
                    }

                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        willChange();
                        setLiner(oldValue);
                        changed();
                        fireFigureHandlesChanged();
                    }

                    @Override
                    public void redo() throws CannotUndoException {
                        super.undo();
                        willChange();
                        setLiner(newValue);
                        changed();
                        fireFigureHandlesChanged();
                    }
                });
            }
        };
        action.putValue(ActionUtil.SELECTED_KEY, getLiner() instanceof ElbowLiner);
        action.putValue(ActionUtil.BUTTON_GROUP_KEY, "liner");
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("liner"));
        actions.add(action);

        return actions;
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        updateLabels();
    }

    private Diagram getDiagram() {
        return (Diagram) getDrawing();
    }

    private Simulation getSimulation() {
        return (getDiagram() == null) ? null : getDiagram().getSimulation();
    }

    @Override
    public RelationshipFigure clone() {
        RelationshipFigure that = (RelationshipFigure) super.clone();
        that.setModel((Relationship) this.getModel().clone());
        return that;
    }

    /**
     * Handles the disconnection of a connection.
     * Override this method to handle this event.
     */
    protected void handleDisconnect(Figure start, Figure end) {
        model.setEnd(null);
        model.setStart(null);
        START_DECORATION.set(this, null);
        END_DECORATION.set(this, null);
    }

    /**
     * Handles the connection of a connection.
     * Override this method to handle this event.
     */
    protected void handleConnect(Figure start, Figure end) {
        if ((start instanceof DiagramFigure) && (end instanceof DiagramFigure)) {
            DiagramFigure sf = (DiagramFigure) start;
            DiagramFigure ef = (DiagramFigure) end;

            model.setEnd((SimulatedElement) ef.getModel());
            model.setStart((SimulatedElement) sf.getModel());

            if (getStartConnector() instanceof RERAttributeConnector) {
                model.setStartAttribute(((RERAttributeConnector) getStartConnector()).getAttribute());
            } else {
                model.setStartAttribute(null);
            }
            if (getEndConnector() instanceof RERAttributeConnector) {
                model.setEndAttribute(((RERAttributeConnector) getEndConnector()).getAttribute());
            } else {
                model.setEndAttribute(null);
            }
        }
    }

    /**
     * Checks if two figures can be connected. Implement this method
     * to constrain the allowed connections between figures.
     */
    public boolean canConnect(Figure start, Figure end) {


        return (start instanceof EntityFigure)
                && (end instanceof EntityFigure);
        /*
        if ((start instanceof DiagramFigure)
        && (end instanceof DiagramFigure)) {
        DiagramFigure sf = (DiagramFigure) start;
        DiagramFigure ef = (DiagramFigure) end;
        return model.canConnect(sf.getModel(), ef.getModel());
        }
        return false;
         */
    }

    public boolean canConnect(Figure start) {
        return start instanceof EntityFigure;
        /*
        if (start instanceof DiagramFigure) {
        DiagramFigure sf = (DiagramFigure) start;
        return model.canConnect(sf.getModel());
        }
        return false;
         */
    }

    public int getConnectionCount() {
        return 0;
    }

    public int getConnectionIndex(DiagramFigure f) {
        return 0;
    }

    @Override
    public void read(DOMInput in) throws IOException {
        readLiner(in);
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setModel((Relationship) in.readObject(0));
        in.closeElement();
        readPoints(in);
        readAttributes(in);
    }

    @Override
    public void write(DOMOutput out) throws IOException {
        out.openElement("Model");
        out.writeObject(getModel());
        out.closeElement();
        writePoints(out);
        writeAttributes(out);
        writeLiner(out);
    }

    @Override
    public int getLayer() {
        return RelationalERDiagram.LINK_LAYER;
    }

    @Override
    public void relationshipAdded(SimulatedObjectEvent e) {
    }

    @Override
    public void relationshipRemoved(SimulatedObjectEvent e) {
    }

    @Override
    public void objectChanged(SimulatedObjectEvent e) {
    }

    @Override
    public void objectAdded(SimulatedObjectEvent e) {
    }

    @Override
    public void objectRemoved(SimulatedObjectEvent e) {
        fireFigureRequestRemove();
    }

    @Override
    public void objectRequestRemove(SimulatedObjectEvent e) {
    }
}

